/*
* avscan.c written by vesely in milan on 28 Nov 2019
* call avfilter scan with hack control file so as to scan given files
*/
/*
Copyright (C) 2019 Alessandro Vesely

This file is part of avfilter

avfilter is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

avfilter is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License version 3
along with avfilter.  If not, see <http://www.gnu.org/licenses/>.
*/

#if defined(HAVE_CONFIG_H)
#include <config.h>
#else
#define PACKAGE_STRING "standalone compile"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <syslog.h>
#include <stdarg.h>
#include <string.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/select.h>
#include <limits.h>
#include <sys/stat.h>
#include "avfiledefs.h"
#include "avfilter_scan.h"

char *argv0; // program name
#if defined __GNUC__
__attribute__ ((format(printf, 2, 3)))
#endif
static void report(int level, char const *fmt, ...)
{
	va_list ap;
	va_start(ap, fmt);
	if (argv0)
	{
		fputs(argv0, stderr);
		fputs(": ", stderr);
		vfprintf(stderr, fmt, ap);
		fputc('\n', stderr);
	}
	else
		vsyslog(level, fmt, ap);
	va_end(ap);
}

static char *mystrerror(int e)
{
	switch (e)
	{
#if EAGAIN == EWOULDBLOCK
		case EAGAIN: return "EAGAIN/EWOULDBLOCK";
#else
		case EAGAIN: return "EAGAIN";
		case EWOULDBLOCK: return "EWOULDBLOCK";
#endif
		case EINPROGRESS: return "EINPROGRESS";
	}
	return "exit";
}

static int open_socket(char *socket_name, struct sockaddr_un *address, int *flags)
/*
* Open a socket and put it in non_blocking mode
*/
{
	int socket_fd = socket(AF_UNIX, SOCK_STREAM, 0);
	if (socket_fd < 0)
	{
		report(LOG_CRIT, "socket failure: %s", strerror(errno));
		return -1;
	}

	memset(address, 0, sizeof *address);
	address->sun_family = AF_UNIX;
	strncpy(address->sun_path, socket_name, sizeof address->sun_path - 1);

	if ((*flags = fcntl(socket_fd, F_GETFL, 0)) == -1) {
		report(LOG_CRIT, "fcntl get failed: %s", strerror(errno));
		close(socket_fd);
		return -1;
	}

	if (fcntl(socket_fd, F_SETFL, *flags | O_NONBLOCK) == -1) {
		report(LOG_CRIT, "set socket_fd nonblocking failed: %s", strerror(errno));
		close(socket_fd);
		return -1;
	}

	return socket_fd;
}

static int av_connect(char *socket_name, struct sockaddr_un *address)
/*
* Open a socket and connect;  address is filled and left there.
*/
{
	int rtc = 0, max_tries = 5;
	int flags;
	int socket_fd = open_socket(socket_name, address, &flags);

	while (socket_fd >= 0 &&
		(rtc = connect(socket_fd, (struct sockaddr *) address, sizeof *address)) != 0 &&
		errno == EAGAIN && max_tries--> 0)
	{
		report(LOG_ERR, "%s not ready to connect (%d tries left)",
			socket_name, max_tries);
		close(socket_fd);
		sleep(1);
		socket_fd = open_socket(socket_name, address, &flags);
	}
	
	if (socket_fd < 0 || (rtc < 0 && errno != EINPROGRESS))
	{
		report(LOG_CRIT, "connect %s returned %d: %s (%s)",
			socket_name, rtc, strerror(errno), mystrerror(errno));
		if (socket_fd >= 0)
			close(socket_fd);
		socket_fd = -1;
	}

	if (socket_fd >= 0 && rtc < 0)  // we're in the backlog, not connected yet
	{
		fd_set set, wset, eset;
		struct timeval timeout;
		timeout.tv_sec = 10;
		timeout.tv_usec = 0;
		FD_ZERO(&set);
		FD_SET(socket_fd, &set);
		wset = set;
		eset = set;
		if ((rtc = select(socket_fd + 1, &set, &wset, &eset, &timeout)) <= 0)
		{
			report(LOG_CRIT, "connection failure: %s",
				rtc < 0? strerror(errno): "timeout");
			close(socket_fd);
			socket_fd = -1;
		}
	}

	int so_error;
	socklen_t len = sizeof so_error;
	if (socket_fd >= 0 &&
		getsockopt(socket_fd, SOL_SOCKET, SO_ERROR, &so_error, &len) < 0)
	{
		report(LOG_CRIT, "getsockopt failed: %s", strerror(errno));
		close(socket_fd);
		socket_fd = -1;
	}

	if (socket_fd >= 0 && so_error != 0)
	{
		report(LOG_CRIT, "SO_ERROR: %d", so_error);
		close(socket_fd);
		socket_fd = -1;
	}

	if (socket_fd >= 0 &&
		fcntl(socket_fd, F_SETFL, flags & ~O_NONBLOCK) == -1)
	{
		report(LOG_CRIT, "set socket_fd non blocking failed: %s", strerror(errno));
		close(socket_fd);
		socket_fd = -1;
	}

	return socket_fd;
}

static int usage(char* arg)
{
	if (arg)
		report(LOG_CRIT, "invalid argument: %s", arg);
	else
		report(LOG_INFO,
			"valid options:\n"
			"\t--help\n"
			"\t--socket socket_name / -s socket_name\n"
			"\t--verbose / -v\n"
			"\t--version\n"
			"\t-l (log to mail log)\n");

	return 2;
}

static char* make_hack_ctlfile(int verbose)
{
	static char hack_ctlfile[] = AVFILTER_TMP_DIR "/hack_ctlXXXXXX";
	int fd = mkstemp(hack_ctlfile);
	FILE *fp = fd >= 0? fdopen(fd, "r+"): NULL;
	if (fd < 0 || fp == NULL)
	{
		report(LOG_CRIT, "cannot create hack_ctl file: %s", strerror(errno));
		return NULL;
	}

	fprintf(fp, "i%s%s\n", hack_uniq, verbose? hack_verbose: "");
	fclose(fp);
	chmod(hack_ctlfile, S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH);
	return hack_ctlfile;
}

int main(int argc, char *argv[])
{
	int sockarg = 0, verbose = 0, i;

	argv0 = strrchr(argv[0], '/');
	if (argv0)
		++argv0;
	else
		argv0 = argv[0];
	

	for (i = 1; i < argc; ++i)
	{
		char *a = argv[i];

		if (a[0] != '-')
			break;
		if (a[1] == '-')
			if (a[2] == 0)
				break;
			else // long opt
			{
				a += 2;
				if (strcmp(a, "help") == 0)
					return usage(NULL);
				else if (strcmp(a, "socket") == 0)
					sockarg = ++i;
				else if (strcmp(a, "verbose") == 0)
					++verbose;
				else if (strcmp(a, "version") == 0)
				{
					puts(PACKAGE_STRING);
					return 2;
				}
				else
					return usage(argv[i]);
			}
		else // short opt
		{
			int ch;
			while ((ch = *(unsigned char*)++a) != 0)
				switch (ch)
				{
					case 'v':
						++verbose;
						break;

					case 's':
						sockarg = ++i;
						break;

					case 'l':
						openlog(argv0, 0, LOG_MAIL);
						argv0 = NULL;
						break;

					default:
						return usage(argv[i]);
				}
		}
	}

	static const char default_socket_name[] = ALLFILTERSOCKETDIR "/avfilter";
	char *socket_name, *free_socket_name = NULL;
	if (sockarg)
	{
		char *sn = argv[sockarg];
		if (sn == NULL)
		{
			report(LOG_CRIT, "missing socket name");
			return 2;
		}

		if (*sn == '/')
			socket_name = sn;
		else
		{
			socket_name = malloc(sizeof default_socket_name + strlen(sn));
			if (socket_name)
			{
				strcpy(socket_name, default_socket_name);
				char *p = strrchr(socket_name, '/');
				if (p)
				{
					strcpy(p + 1, sn);
					free_socket_name = socket_name;
				}
				else
				{
					free(socket_name);
					socket_name = NULL;
				}
			}
		}
	}
	else
		socket_name = (char*)default_socket_name;

	if (socket_name == NULL)
	{
		report(LOG_CRIT, "failure");
		return 2;
	}

	int rtc = 0;
	char *hack_ctlfile = make_hack_ctlfile(verbose);
	if (hack_ctlfile == NULL)
	{
		free(free_socket_name);
		return 2;
	}

	for (; i < argc; ++i) // usually one arg
	{
		size_t path_max;
#ifdef PATH_MAX
		path_max = PATH_MAX;
#else
		path_max = pathconf(path, _PC_PATH_MAX);
#endif
		if (path_max <= 4096)
			path_max = 4096;
		char buf[path_max];
		if (realpath(argv[i], buf) == NULL)
		{
			report(LOG_ERR, "cannot resolve %s: %s",
				argv[i], strerror(errno));
			continue;
		}

		struct sockaddr_un address;
		int socket_fd = av_connect(socket_name, &address);
		FILE *socket_fp = socket_fd >= 0? fdopen(socket_fd, "r+"): NULL;
		if (socket_fd < 0 || socket_fp == NULL)
		{
			rtc = 2;
			break;
		}

		fprintf(socket_fp, "%s\n%s\n\n", buf, hack_ctlfile);
		fflush(socket_fp);

		char *s = fgets(buf, sizeof buf, socket_fp);
		if (s)
		{
			char *t = NULL;
			unsigned long frtc = strtoul(s, &t, 10);
			if (frtc < INT_MAX && t && *t == '\n')
			{
				*t = 0;
				if (verbose)
					printf("%s: %s\n", argv[i],
						frtc == 0? "OK": frtc == 1? "VIRUS": "FATAL");
				if (frtc > 1)
					report(LOG_ERR, "%s returned %s for %s\n",
						socket_name, s, argv[i]);
				else if (rtc < (int)frtc)
					rtc = (int)frtc;
				if (verbose)
					while ((s = fgets(buf, sizeof buf, socket_fp)) != NULL)
						fputs(s, stdout);
			}
			else
			{
				report(LOG_CRIT, "bad reply from %s: %s",
					socket_name, s);
				rtc = 2;
			}
		}
		else
		{
			report(LOG_CRIT, "no reply from %s: %s",
				socket_name, strerror(errno));
			rtc = 2;
		}
		fclose(socket_fp);
	}

	unlink(hack_ctlfile);
	free(free_socket_name);
	return rtc;
}
