/*
* parm.c - copied from zdkimfilter/src by ale in milano on 24jun2014
* parameter file parsing

Copyright (C) 2014 Alessandro Vesely

This file is part of avfilter

avfilter is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

avfilter is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License version 3
along with avfilter.  If not, see <http://www.gnu.org/licenses/>.

*/
#include <config.h>
#if TEST_AVFILTER
#define AVFILTER_DEBUG 1
#endif
#if !AVFILTER_DEBUG && !NDEBUG
#define NDEBUG
#endif
#include <inttypes.h>
#include <string.h>
#include <syslog.h>
#include <errno.h>
#include <limits.h>
#include <ctype.h>
#include <stddef.h>
#include <stdarg.h>
#include "parm.h"
#include "avfiledefs.h"
#include "vb_fgets.h"
#include "log.h"

#include <clamav.h>

#include <assert.h>

/*
* logging weariness:
*/

static stdlog_t do_report = &stdlog;

/*************************************************************************
** configuration parameters
*************************************************************************/

typedef struct config_conf
{
	char const *name, *descr;
	int (*assign_fn)(parm_t*, struct config_conf const*, char*);
	unsigned flag;
	size_t offset, size;
} config_conf;

// access members of variable parm (type parm_t), left and right hand
#define PARM_PTR(T) *(T*)(((char*)parm) + c->offset)

static int add_freable(parm_t *parm, void *v)
{
	malloced_parm *m = malloc(sizeof(malloced_parm));
	if (m == NULL)
	{
		(*do_report)(LOG_ALERT, "MEMORY FAULT");
		return -1;
	}

	m->malloced = v;
	m->next = parm->freable;
	parm->freable = m;
	return 0;
}

static void sub_freable(parm_t *parm, void *v)
{
	malloced_parm **pm = &parm->freable;

	while (*pm)
	{
		if ((*pm)->malloced == v)
		{
			void *tmp = *pm;
			*pm = (*pm)->next;
			free(tmp);
			break;
		}
	}
}

static int
assign_string(parm_t *parm, config_conf const *c, char*s, int lastchar)
{
	assert(parm && c && s && c->size == sizeof(char*));
	size_t len = strlen(s);
	if (len > 1 && s[len-1] == lastchar)
		--len;

	char *v = malloc(len + 1);
	if (v == NULL)
	{
		(*do_report)(LOG_ALERT, "MEMORY FAULT");
		return -1;
	}

	memcpy(v, s, len);
	v[len] = 0;

	PARM_PTR(char*) = v;
	return add_freable(parm, v);
}

static int assign_dir(parm_t *parm, config_conf const *c, char*s)
{
	return assign_string(parm, c, s, '/');
}

static int assign_header(parm_t *parm, config_conf const *c, char*s)
{
	return assign_string(parm, c, s, ':');
}

static int
assign_char(parm_t *parm, config_conf const *c, char*s)
{
	assert(parm && c && s && c->size == sizeof(char));
	char ch = *s, v;
	if (strchr("YyTt1", ch)) v = 1; //incl. ch == 0
	else if (strchr("Nn0", ch)) v = 0;
	else return -1;
	PARM_PTR(char) = v;
	return 0;
}

static int
assign_option(parm_t *parm, config_conf const *c, char*s)
{
	assert(parm && c && s && c->size == sizeof(unsigned int) &&
		(c->offset == offsetof(parm_t, scan_options.general) ||
		c->offset == offsetof(parm_t, scan_options.parse) ||
		c->offset == offsetof(parm_t, scan_options.heuristic) ||
		c->offset == offsetof(parm_t, scan_options.mail) ||
		c->offset == offsetof(parm_t, scan_options.dev) ||
		c->offset == offsetof(parm_t, load_options)));

	char ch = *s;
	unsigned int options = PARM_PTR(unsigned int);
	if (c->flag)
	{
		if (strchr("YyTt1", ch)) options |= c->flag; //incl. ch == 0
		else if (strchr("Nn0", ch)) options &= ~ c->flag;
		else return -1;
	}
	else // structured-ssn-format
	{
		assert(c->offset == offsetof(parm_t, scan_options.heuristic));

		switch (ch)
		{
			case '0':
				options |= CL_SCAN_HEURISTIC_STRUCTURED_SSN_NORMAL;
				break;
			case '1':
				options |= CL_SCAN_HEURISTIC_STRUCTURED_SSN_STRIPPED;
				break;
			case '2':
				options |=
					(CL_SCAN_HEURISTIC_STRUCTURED_SSN_NORMAL |
						CL_SCAN_HEURISTIC_STRUCTURED_SSN_STRIPPED);
				break;
			default:
				return -1;
		}
	}
	PARM_PTR(unsigned int) = options;
	return 0;
}

static int
do_assign_int(parm_t *parm, config_conf const *c, uintmax_t l)
{
	size_t const size = c->size;
	int err = 0;
	switch (size)
	{
		case sizeof(uint8_t):
			if (l > UINT8_MAX) err = errno = ERANGE;
			else PARM_PTR(uint8_t) = (uint8_t)l;
			break;
		case sizeof(uint16_t):
			if (l > UINT16_MAX) err = errno = ERANGE;
			else PARM_PTR(uint16_t) = (uint16_t)l;
			break;
		case sizeof(uint32_t):
			if (l > UINT32_MAX) err = errno = ERANGE;
			else PARM_PTR(uint32_t) = (uint32_t)l;
			break;
		case sizeof(uint64_t):
			if (l > UINT64_MAX) err = errno = ERANGE;
			else PARM_PTR(uint64_t) = (uint64_t)l;
			break;
		default:
			err = errno = ERANGE;
			assert(0);
			break;
	}
	return err? -1: 0;
}

static int
assign_int(parm_t *parm, config_conf const *c, char*s)
{
	assert(parm && c && s && c->size > 0 && c->size <= sizeof(uintmax_t));
	char *t = NULL;
	errno = 0;
	uintmax_t l = strtoimax(s, &t, 0);
	if (errno != ERANGE && t && *t == 0)
		return do_assign_int(parm, c, l);

	return -1;
}

static int
assign_size(parm_t *parm, config_conf const *c, char*s)
{
	assert(parm && c && s && c->size == sizeof(uint64_t));
	char *t = NULL;
	errno = 0;
	uintmax_t l = strtoimax(s, &t, 0);
	if (errno != ERANGE && l < UINT_MAX && t)
	{
		unsigned long k = 1UL;
		switch (*t)
		{
			case 0:
				break;

#pragma GCC diagnostic push
#pragma GCC diagnostic ignored "-Wimplicit-fallthrough="
			case 'M':
			case 'm':
				k = 1024UL;
				// through
			case 'K':
			case 'k':
				k *= 1024UL;
				++t;
				break;
#pragma GCC diagnostic pop

			default:
				return -1;
		}
		if (l < UINT_MAX/k) l *= k;
		else return -1;
	}
	else return -1;
	// if (*t) return -1;

	return do_assign_int(parm, c, l);
}

static void hfields(char *h, charray *sz)
{
	assert(h && sz);
	assert(sz->alloc == 0 || sz->count <= sz->alloc);
	assert(sz->alloc == 0 || sz->strsize <= sz->stralloc);
	assert(sz->alloc == 0 || sz->count == 0 ||
		sz->array[sz->count - 1] + strlen(sz->array[sz->count - 1]) + 1 ==
			CHARRAY_FIRST_STRING(*sz) + sz->strsize);

	char *s = h, *str = NULL, **a = NULL;
	size_t count = 0, strsize = 0;
	int ch;

	if (sz->alloc)
	{
		a = &sz->array[sz->count];
		str = CHARRAY_FIRST_STRING(*sz) + sz->strsize;
	}

	for (;;)
	{
		while (isspace(ch = *(unsigned char*)s))
			++s;
		if (ch == 0)
			break;

		char *field = s;
		++count;
		++s;
		while (!isspace(ch = *(unsigned char*)s) && ch != 0)
			++s;

		size_t const l = s - field;
		strsize += l + 1;

		if (a)
		{
			memcpy(str, field, l);
			*a++ = str;
			str += l;
			*str++ = 0;

			assert(str <= CHARRAY_OFF_THE_END(*sz));
		}
		if (ch == 0)
			break;
	}

	sz->count += count;
	sz->strsize += strsize;
	assert(sz->alloc == 0 || sz->count <= sz->alloc);
	assert(sz->alloc == 0 || sz->strsize <= sz->stralloc);
	assert(sz->alloc == 0 || sz->count == 0 ||
	sz->array[sz->count - 1] + strlen(sz->array[sz->count - 1]) + 1 ==
		CHARRAY_FIRST_STRING(*sz) + sz->strsize);
}

static int
assign_array(parm_t *parm, config_conf const *c, char*s)
{
	assert(parm && c && s && c->size == sizeof(charray*));

	charray *a = PARM_PTR(charray*), sz;
	memset(&sz, 0, sizeof sz);
	hfields(s, &sz);
	if (sz.count > 0)
	{
		if (a)
		{
			size_t need_count = a->count + sz.count,
				need_strsize = a->strsize + sz.strsize ;

			if (need_count > a->alloc || need_strsize > a->stralloc) // realloc
			{
				need_count *= 2;
				need_strsize *= 2;
				if (need_count < a->alloc) need_count = a->alloc;
				else if (need_strsize < a->stralloc) need_strsize = a->stralloc;

				size_t const need_total = need_strsize + need_count* sizeof(char*);
				char* const first_a = CHARRAY_FIRST_STRING(*a);
				assert(need_total > a->stralloc + a->alloc*sizeof(char*));

				charray *b = realloc(a, need_total + sizeof(charray));
				if (b == NULL)
				{
					(*do_report)(LOG_ALERT, "MEMORY FAULT");
					return -1;
				}

#pragma GCC diagnostic push
#pragma GCC diagnostic ignored "-Wuse-after-free"
				b->alloc = need_count; // used in CHARRAY_FIRST_STRING
				b->stralloc = need_strsize;

				char* const first_b = CHARRAY_FIRST_STRING(*b);
				ptrdiff_t const diff = first_b - first_a;
				if (diff)
				{
					memmove(first_b, first_a + ((char*)b - (char*)a), b->strsize);
					for (size_t i = 0; i < b->count; ++i)
						b->array[i] += diff;
				}

				if (a != b)
				{
					sub_freable(parm, a);
					a = b;
					if (add_freable(parm, a))
						return -1;
				}
#pragma GCC diagnostic pop
			}
		}
		else // new
		{
			// the array size in the definition of charray is wasted, as well as rounding
			sz.stralloc = 3 * sz.strsize;
			sz.alloc = 3 * sz.count + 1;
			a = (charray*)calloc(1,
				sz.stralloc + sz.alloc*sizeof(char*) + sizeof(charray));
			if (a == NULL || add_freable(parm, a) != 0)
			{
				(*do_report)(LOG_ALERT, "MEMORY FAULT");
				return -1;
			}

			a->alloc = sz.alloc;
			a->stralloc = sz.stralloc;
		}

		hfields(s, a);
	}

	PARM_PTR(charray*) = a;
	return 0;
}

static int
assign_case(parm_t *parm, config_conf const *c, char*s)
{
	assert(parm && c && s &&
		((c->size == sizeof parm->bytecode_mode &&
			c->offset == offsetof(parm_t, bytecode_mode)) ||
		(c->size	== sizeof parm->action_default &&
			c->offset == offsetof(parm_t, action_default))));

	if (c->offset == offsetof(parm_t, bytecode_mode))
	{
		if (strcmp(s, "ForceJIT") == 0)
			parm->bytecode_mode = CL_BYTECODE_MODE_JIT;
		else if (strcmp(s, "ForceInterpreter") == 0)
			parm->bytecode_mode = CL_BYTECODE_MODE_INTERPRETER;
		else if (strcmp(s, "Test") == 0)
			parm->bytecode_mode = CL_BYTECODE_MODE_TEST;
		else if (strcmp(s, "Auto") == 0)
			parm->bytecode_mode = CL_BYTECODE_MODE_AUTO;
		else
			return -1;
	}
	else
	{
		if (strcmp(s, "pass") == 0)
			parm->action_default = msg_pass;
		else if (strcmp(s, "reject") == 0)
			parm->action_default = msg_reject;
		else if (strcmp(s, "drop") == 0)
			parm->action_default = msg_drop;
		else
			return -1;
	}

	return 0;
}

#define STRING2(P) #P
#define STRING(P) STRING2(P)
#define CONFIG(P,D,F,B) {STRING(P), D, F, B, \
	offsetof(parm_t, P), sizeof(((parm_t*)0)->P) }

// scan (CL_SCAN_*) AND load (CL_DB_*) have a flag, and fn==assign_option!
#define CONFIG_SCAN_GENERAL(P,D,B) {STRING(P), D, assign_option, B, \
	offsetof(parm_t, scan_options.general), sizeof(((parm_t*)0)->scan_options.general) }
#define CONFIG_SCAN_PARSE(P,D,B) {STRING(P), D, assign_option, B, \
	offsetof(parm_t, scan_options.parse), sizeof(((parm_t*)0)->scan_options.parse) }
#define CONFIG_SCAN_HEURISTIC(P,D,B) {STRING(P), D, assign_option, B, \
	offsetof(parm_t, scan_options.heuristic), sizeof(((parm_t*)0)->scan_options.heuristic) }
#define CONFIG_SCAN_MAIL(P,D,B) {STRING(P), D, assign_option, B, \
	offsetof(parm_t, scan_options.mail), sizeof(((parm_t*)0)->scan_options.mail) }
#define CONFIG_SCAN_DEV(P,D,B) {STRING(P), D, assign_option, B, \
	offsetof(parm_t, scan_options.dev), sizeof(((parm_t*)0)->scan_options.dev) }
#define CONFIG_LOAD(P,D,B) {STRING(P), D, assign_option, B, \
	offsetof(parm_t, load_options), sizeof(((parm_t*)0)->load_options) }

#define NO_FLAG UINT_MAX


/*
* each option has to be defined in three places: 1 parm_t definition in parm.h,
* 2 the conf[] array below, and 3 etc/avfilter.conf.pod.in
*
* The order here reflects how the data is displayed by reporting functions.
*
* The values are also used for making assignments after reading the
* config file.  The "assign" functions defined above take a parm_t
* object into which to do the assignment.
*
* Print functions below use these values to display settings.
*/

static config_conf const conf[] =
{
	CONFIG(all_mode, "bool", assign_char, NO_FLAG),

	CONFIG(verbose, "int", assign_int, NO_FLAG),

	/* directories */
	CONFIG(save_virus, "directory", assign_dir, NO_FLAG),
	CONFIG(save_only_if_drop, "bool", assign_char, NO_FLAG),
	CONFIG(save_missed, "directory", assign_dir, NO_FLAG),
	CONFIG(tempdir, "directory", assign_dir, CL_ENGINE_TMPDIR),
	CONFIG(database, "directory, load", assign_dir, NO_FLAG),
	CONFIG(piddir, "directory", assign_dir, NO_FLAG),

	/* actions */
	CONFIG(action_default, "pass/reject/drop", assign_case, NO_FLAG),
	CONFIG(virus_header, "string", assign_header, NO_FLAG),
	CONFIG(pass_recipient, "array", assign_array, NO_FLAG),
	CONFIG(always_pass, "array", assign_array, NO_FLAG),
	CONFIG(pass, "array", assign_array, NO_FLAG),
	CONFIG(reject, "array", assign_array, NO_FLAG),
	CONFIG(drop, "array", assign_array, NO_FLAG),

	/*
	* dboptions for cl_load:
	* name is like clamscan (but lowcase, underscore),
	* flag is the bit as defined in clamav.h CL_DB_*
	*/
	CONFIG_LOAD(phishing_sigs, "bool", CL_DB_PHISHING),
	CONFIG_LOAD(official_db_only, "bool", CL_DB_OFFICIAL_ONLY),
	CONFIG_LOAD(phishing_scan_urls, "bool", CL_DB_PHISHING_URLS),
	CONFIG_LOAD(bytecode, "bool", CL_DB_BYTECODE),
	CONFIG_LOAD(detect_pua, "bool", CL_DB_PUA),
	CONFIG(pua, "array", assign_array, CL_ENGINE_PUA_CATEGORIES),
	CONFIG_LOAD(exclude_pua, "bool", CL_DB_PUA_EXCLUDE),
	CONFIG_LOAD(include_pua, "bool", CL_DB_PUA_INCLUDE),
	CONFIG_LOAD(bytecode_unsigned, "bool", CL_DB_BYTECODE_UNSIGNED),

	/*
	* engine options for cl_engine_set/get_num:
	* name is like clamscan (but lowcase, underscore),
	* flag is the enum defined in clamav.h CL_ENGINE_*
	*/

	CONFIG(dev_ac_only, "bool", assign_char, CL_ENGINE_AC_ONLY),
	CONFIG(dev_ac_depth, "int", assign_int, CL_ENGINE_AC_MAXDEPTH),
	CONFIG(leave_temps, "bool", assign_char, CL_ENGINE_KEEPTMP),
	CONFIG(force_to_disk, "bool", assign_char, CL_ENGINE_FORCETODISK),
	CONFIG(bytecode_statistics, "bool", assign_char, CL_ENGINE_AC_ONLY),
	CONFIG(bytecode_timeout, "int", assign_int, CL_ENGINE_BYTECODE_TIMEOUT),
	CONFIG(bytecode_mode, "ForceJIT/ForceInterpreter/Test/Auto", assign_case,
		CL_ENGINE_BYTECODE_MODE),
	CONFIG(max_scansize, "size", assign_size, CL_ENGINE_MAX_SCANSIZE),
	CONFIG(max_filesize, "size", assign_size, CL_ENGINE_MAX_FILESIZE),
	CONFIG(max_files, "size", assign_size, CL_ENGINE_MAX_FILES),
	CONFIG(max_recursion, "int", assign_int, CL_ENGINE_MAX_RECURSION),
	CONFIG(max_embeddedpe, "size", assign_size, CL_ENGINE_MAX_EMBEDDEDPE),
	CONFIG(max_htmlnormalize, "size", assign_size, CL_ENGINE_MAX_HTMLNORMALIZE),
	CONFIG(max_htmlnotags, "size", assign_size, CL_ENGINE_MAX_HTMLNOTAGS),
	CONFIG(max_scriptnormalize, "size", assign_size,
		CL_ENGINE_MAX_SCRIPTNORMALIZE),
	CONFIG(max_ziptypercg, "size", assign_size, CL_ENGINE_MAX_ZIPTYPERCG),
	/* TODO: version 101.4 and better, with long long max_scantime
	CONFIG(max_scantime, "millisecs", assign_int, CL_ENGINE_MAX_ZIPTYPERCG),
	*/

	/*
	* scan options for cl_scandesc/file*:
	* name is like clamscan (but lowcase, underscore),
	* flag is the bit as defined in clamav.h CL_SCAN_*
	*/
	CONFIG_SCAN_GENERAL(allmatch, "bool", CL_SCAN_GENERAL_ALLMATCHES),
	CONFIG_SCAN_GENERAL(heuristic_scan_precedence, "bool", CL_SCAN_GENERAL_HEURISTIC_PRECEDENCE),
	CONFIG_SCAN_PARSE(scan_archive, "bool", CL_SCAN_PARSE_ARCHIVE),

	CONFIG_SCAN_HEURISTIC(alert_broken, "bool", CL_SCAN_HEURISTIC_BROKEN),
	CONFIG_SCAN_HEURISTIC(alert_exceeds_max, "bool", CL_SCAN_HEURISTIC_EXCEEDS_MAX),
	CONFIG_SCAN_HEURISTIC(alert_phishing_ssl, "bool", CL_SCAN_HEURISTIC_PHISHING_SSL_MISMATCH),
	CONFIG_SCAN_HEURISTIC(alert_phishing_cloak, "bool", CL_SCAN_HEURISTIC_PHISHING_CLOAK),
	CONFIG_SCAN_HEURISTIC(alert_macros, "bool", CL_SCAN_HEURISTIC_MACROS),
	CONFIG_SCAN_HEURISTIC(alert_encrypted, "bool", CL_SCAN_HEURISTIC_ENCRYPTED_ARCHIVE | CL_SCAN_HEURISTIC_ENCRYPTED_DOC),
	CONFIG_SCAN_HEURISTIC(alert_encrypted_archive, "bool", CL_SCAN_HEURISTIC_ENCRYPTED_ARCHIVE),
	CONFIG_SCAN_HEURISTIC(alert_encrypted_doc, "bool", CL_SCAN_HEURISTIC_ENCRYPTED_DOC),
	CONFIG_SCAN_HEURISTIC(alert_partition_intersection, "bool", CL_SCAN_HEURISTIC_PARTITION_INTXN),
	CONFIG_SCAN_HEURISTIC(alert_broken_media, "bool", CL_SCAN_HEURISTIC_BROKEN_MEDIA),

	CONFIG_SCAN_PARSE(scan_pe, "bool", CL_SCAN_PARSE_PE),
	CONFIG_SCAN_PARSE(scan_elf, "bool", CL_SCAN_PARSE_ELF),
	CONFIG_SCAN_PARSE(scan_ole2, "bool", CL_SCAN_PARSE_OLE2),
	CONFIG_SCAN_PARSE(scan_pdf, "bool", CL_SCAN_PARSE_PDF),
	CONFIG_SCAN_PARSE(scan_swf, "bool", CL_SCAN_PARSE_SWF),
	CONFIG_SCAN_PARSE(scan_html, "bool", CL_SCAN_PARSE_HTML),
	CONFIG_SCAN_PARSE(scan_mail, "bool", CL_SCAN_PARSE_MAIL),
	CONFIG_SCAN_GENERAL(heuristic_alerts, "bool", CL_SCAN_GENERAL_HEURISTICS),
	// not in man CONFIG_SCAN(dev_performance, "bool", CL_SCAN_DEV_COLLECT_PERFORMANCE_INFO),
	CONFIG_SCAN_HEURISTIC(detect_structured, "bool", CL_SCAN_HEURISTIC_STRUCTURED),
	CONFIG_SCAN_HEURISTIC(structured_ssn_format, "0/1/2", 0 /* special case */),

	// moved downward
	CONFIG(structured_ssn_count, "int", assign_int, CL_ENGINE_MIN_SSN_COUNT),
	CONFIG(structured_cc_count, "int", assign_int, CL_ENGINE_MIN_CC_COUNT),

	/* sentinel */
	{NULL, NULL, NULL, 0, 0, 0}
};


// compile time check: is_defined must have a bit for each configured parameter
// --its typedef in parm.h has to be increased if this fails.
// $ echo $(egrep '^[[:space:]]*CONFIG' parm.c |wc -l)/8 |bc -l
static_assert(sizeof(is_defined_bitflag)*8 >= sizeof conf/sizeof conf[0]);

static inline int get_defined(is_defined_bitflag const is_defined, size_t i)
{
	div_t ndx = div(i, 8);
	return (is_defined[ndx.quot] & 1 << ndx.rem) != 0;
}

static inline void set_defined(is_defined_bitflag is_defined, size_t i)
{
	div_t ndx = div(i, 8);
	is_defined[ndx.quot] |= 1 << ndx.rem;
}

void clear_parm(parm_t *parm)
{
	assert(parm);

	for (malloced_parm *m = parm->freable; m;)
	{
		malloced_parm *tmp = m->next;
		free(m->malloced);
		free(m);
		m = tmp;
	}
}

static config_conf const *conf_name(char const *p)
{
	for (config_conf const *c = conf; c->name; ++c)
		if (strcmp(c->name, p) == 0)
			return c;

	return NULL;
}

int read_all_values(parm_t *parm, char const *fname)
// initialization, 0 on success
{
	assert(parm);
	assert(fname);

	int line_no = 0;
	errno = 0;

	FILE *fp = fopen(fname, "r");
	if (fp == NULL)
	{
		(*do_report)(LOG_ALERT,
			"Cannot read %s: %s", fname, strerror(errno));
		return -1;
	}

	var_buf vb;
	if (vb_init(&vb))
	{
		fclose(fp);
		return -1;
	}

	int errs = 0;
	size_t keep = 0;
	char *p;

	while ((p = vb_fgets(&vb, keep, fp)) != NULL)
	{
		char *eol = p + strlen(p) - 1;
		int ch = 0;
		++line_no;

		while (eol >= p && isspace(ch = *(unsigned char*)eol))
			*eol-- = 0;

		if (ch == '\\')
		{
			*eol = ' '; // this replaces the backslash
			keep += eol + 1 - p;
			continue;
		}

		/*
		* full logic line
		*/
		keep = 0;

		char *s = p = vb.buf;
		while (isspace(ch = *(unsigned char*)s))
			++s;
		if (ch == '#' || ch == 0)
			continue;

		char *const name = s;
		while (isalnum(ch = *(unsigned char*)s) || ch == '_')
			++s;
		*s = 0;
		config_conf const *c = conf_name(name);
		if (c == NULL)
		{
			(*do_report)(LOG_ERR,
				"Invalid name %s at line %d in %s", name, line_no, fname);
			++errs;
			continue;
		}

		*s = ch;
		while (isspace(ch = *(unsigned char*)s) || ch == '=')
			++s;

		char *const value = s;

		if ((*c->assign_fn)(parm, c, value) == 0)
		{
			size_t const ndx = c - &conf[0];
			set_defined(parm->is_defined, ndx);
		}
		else
		{
			(*do_report)(LOG_ERR,
				"Invalid value %s for %s at line %d in %s",
					value, c->name, line_no, fname);
			++errs;
		}
	}

	if (errs)
		(*do_report)(LOG_ALERT, "%d error(s) in %s", errs, fname);

	vb_clean(&vb);
	fclose(fp);

	return errs;
}

static char *get_dir(parm_t const *parm, config_conf const *c)
{
	assert(parm && c);
	assert(c->assign_fn == assign_dir);

	return PARM_PTR(char*);
}

static char *get_directory(parm_t const *parm, char const *name)
{
	assert(parm);
	assert(name);

	unsigned int name_ndx;
	for (name_ndx = 0;; ++name_ndx)
	{
		if (conf[name_ndx].name == NULL)
			return NULL;

		if (strcmp(conf[name_ndx].name, name) == 0)
			break;
	}

	if (conf[name_ndx].assign_fn != assign_dir)
		return NULL;

	return get_dir(parm, &conf[name_ndx]);
}

char *read_directory(char const *fname, char const *name)
/*
* Read a string named name from parameter file and return malloced string.
* return NULL for fname problem, memory fault, invalid parameter.

* Only string values are supported.
* Only values in parm_t are supported.
*/
{
	assert(fname);
	assert(name);

	parm_t *parm = calloc(1, sizeof *parm);
	if (parm == NULL)
		return NULL;

	int errs = read_all_values(parm, fname);
	if (errs)
	{
		free(parm);
		return NULL;
	}

	char *dir = get_directory(parm, name);
	if (dir)
		dir = strdup(dir);

	clear_parm(parm);
	free(parm);
	return dir;
}

#if defined TEST_DIRECTORY
#define NO_CLAMAV_FUNCTIONS
#endif

#if !defined NO_CLAMAV_FUNCTIONS
//  the version without clamav functions is linked with avfilter_sig

static uintmax_t get_int(parm_t const *parm, config_conf const *c)
{
	assert(parm && c);
	assert(c->assign_fn == assign_int || c->assign_fn == assign_size);

	size_t const size = c->size;
	switch (size)
	{
		case sizeof(uint8_t):
			return PARM_PTR(uint8_t);

		case sizeof(uint16_t):
			return PARM_PTR(uint16_t);

		case sizeof(uint32_t):
			return PARM_PTR(uint32_t);

		case sizeof(uint64_t):
			return PARM_PTR(uint64_t);

		default:
			assert(0);
			return 0;
	}
}

static int
get_bool(parm_t const *parm, config_conf const *c, char const **at, int *def)
// char and options
{
	assert(parm && c);
	assert(
		(c->size == sizeof(unsigned int) &&
			(
				c->offset == offsetof(parm_t, scan_options.general) ||
				c->offset == offsetof(parm_t, scan_options.parse) ||
				c->offset == offsetof(parm_t, scan_options.heuristic) ||
				c->offset == offsetof(parm_t, scan_options.mail) ||
				c->offset == offsetof(parm_t, scan_options.dev) ||
				c->offset == offsetof(parm_t, load_options)
			)
		) ||
		(c->size == 1 && c->assign_fn == assign_char));
	assert(at);
	assert(def);

	if (c->size == 1)
	{
		return PARM_PTR(char) != 0;
	}
	else if (c->flag)
	{
		switch (c->offset)
		{
			case offsetof(parm_t, scan_options.general):
				*at = "scan";
				*def = 0;
				break;

			case offsetof(parm_t, scan_options.parse):
				*at = "scan";
				*def = 1;  // all parsers enabled by default
				break;

			case offsetof(parm_t, scan_options.heuristic):
				*at = "scan";
				*def = (c->flag & CL_SCAN_GENERAL_HEURISTICS) != 0;
				break;

			case offsetof(parm_t, scan_options.mail):
				*at = "scan";
				*def = (c->flag & CL_SCAN_MAIL_PARTIAL_MESSAGE) != 0;
				break;

			case offsetof(parm_t, scan_options.dev):
				*at = "scan";
				*def = 0;
				break;

			case offsetof(parm_t, load_options):
				*at = "load";
				*def = 0;
				break;

			default:
				*at = "ERROR: CORRUPTED CONFIG CONSTANT";
				*def = 0;
				assert(0);
				break;
		}

		return (PARM_PTR(unsigned int) & c->flag) != 0;
	}
	else // structured-ssn-format
	{
		assert(c->offset == offsetof(parm_t, scan_options.heuristic));
		unsigned int const u = parm->scan_options.heuristic &
			(CL_SCAN_HEURISTIC_STRUCTURED_SSN_NORMAL |
				CL_SCAN_HEURISTIC_STRUCTURED_SSN_STRIPPED);

		*at = "scan";
		*def = 0;

		switch (u)
		{
			case CL_SCAN_HEURISTIC_STRUCTURED_SSN_NORMAL:
				return 0;

			case CL_SCAN_HEURISTIC_STRUCTURED_SSN_STRIPPED:
				return 1;

			case CL_SCAN_HEURISTIC_STRUCTURED_SSN_NORMAL |
					CL_SCAN_HEURISTIC_STRUCTURED_SSN_STRIPPED:
				return 2;

			default:
				assert(u == 0); // not set
				return -1;
		}
	}
}

static char const *get_case(parm_t const *parm, config_conf const *c)
{
	assert(parm && c && c->size == sizeof(uint8_t) &&
		(c->offset == offsetof(parm_t, bytecode_mode) ||
			c->offset == offsetof(parm_t, action_default)));

	if (c->offset == offsetof(parm_t, bytecode_mode))
	{
		switch (parm->bytecode_mode)
		{
			case CL_BYTECODE_MODE_JIT: return "ForceJIT";
			case CL_BYTECODE_MODE_INTERPRETER: return "ForceInterpreter";
			case CL_BYTECODE_MODE_TEST: return "Test";
			case CL_BYTECODE_MODE_AUTO: return "Auto";
			default: break;
		}
	}
	else
	{
		switch (parm->action_default)
		{
			case msg_pass: return "pass";
			case msg_reject: return "reject";
			case msg_drop: return "drop";
			default: break;
		}
	}

	assert(0);
	return " ------> INVALID!!";
}

void apply_parm_defaults(parm_t *parm)
/*
* Set values to undefined parameters that deserve a default.
* ClamAV has its own defaults, this function must be called after parm
* has been read from file, but before setting/using values.
*/
{
	if (parm->tempdir == NULL)
		parm->tempdir = AVFILTER_TMP_DIR;

	if (parm->database == NULL)
		parm->database = (char *)cl_retdbdir();

	if (parm->load_options == 0)
		parm->load_options = CL_DB_STDOPT;

	if (parm->scan_options.general == 0 &&
		parm->scan_options.parse == 0 &&
		parm->scan_options.heuristic == 0 &&
		parm->scan_options.mail == 0 &&
		parm->scan_options.dev == 0)
	{
		parm->scan_options.parse |= ~0; /* enable all parsers */
		parm->scan_options.general |= CL_SCAN_GENERAL_HEURISTICS; /* enable heuristic alert options */
		parm->scan_options.mail |= CL_SCAN_MAIL_PARTIAL_MESSAGE; /* no easy attack paths... */
	}

	parm->applied_parm_defaults = 1;
}

int set_engine_values(parm_t const *parm, struct cl_engine *clamav)
{
	assert(parm);
	int err = 0;

	config_conf const *c = &conf[0];
	for (size_t i = 0; c->name; ++c, ++i)
	{
#if defined TEST_AVFILTER
		if (parm->verbose >= 9)
			(*do_report)(LOG_DEBUG,
				"%sefined engine value %s, flag=0x%x, assign_fn=%p",
					get_defined(parm->is_defined, i)? "D": "Not d",
					c->name, c->flag, c->assign_fn);
#endif

		// CL_ENGINE_* are the only flagged values individually assigned
		if (c->flag != NO_FLAG && c->assign_fn != assign_option &&
			get_defined(parm->is_defined, i))
		{
#if defined TEST_ARRAY
			(void)clamav; // not linked with libclamav
#else
			int rtc;
			if (c->offset == offsetof(parm_t, tempdir))
			{
				rtc = cl_engine_set_str(clamav, c->flag, parm->tempdir);
			}
			else if (c->offset == offsetof(parm_t, pua))
			{
				assert(parm->pua);

				charray const* const pua = parm->pua;
				char *pua_cat = malloc(pua->strsize + 1);
				if (pua_cat == NULL)
				{
					(*do_report)(LOG_ALERT, "MEMORY FAULT");
					rtc = 1;
				}
				else
				{
					size_t off = 0;
					for (size_t i = 0; i < pua->count; ++i)
					{
						char const *const cat = pua->array[i];
						size_t len = strlen(cat);
						memcpy(&pua_cat[off], cat, len);
						off += len;
						pua_cat[off++] = '.';
					}
					assert(off == pua->strsize);
					pua_cat[off] = 0;
					rtc = cl_engine_set_str(clamav, c->flag, pua_cat);
					free(pua_cat);
				}
			}
			else if (c->assign_fn == assign_char)
			{
				char const *dummy1;
				int dummy2;
				rtc = cl_engine_set_num(clamav, c->flag,
					get_bool(parm, c, &dummy1, &dummy2));
			}
			else
			{
				assert(c->assign_fn == assign_int ||
					c->assign_fn == assign_size);
				rtc = cl_engine_set_num(clamav, c->flag, get_int(parm, c));
			}

			if (rtc)
			{
				if (parm->verbose)
					(*do_report)(LOG_ERR,
						"cannot set %s; engine: %s",
							c->name, cl_strerror(rtc));
				err = 1;
			}
#endif
		}
	}

	return err;
}

static const char*get_default_dir(size_t offset)
{
	char const *default_desc = NULL;

	switch (offset)
	{
		case offsetof(parm_t, tempdir):
			default_desc = AVFILTER_TMP_DIR;
			break;

		case offsetof(parm_t, database):
			default_desc = cl_retdbdir();
			break;

		default: // save_* are disabled by default
			break;
	}

	return default_desc;
}

void print_parm(parm_t const *parm)
{
	assert(parm);

	config_conf const *c = &conf[0];
	size_t ndx = 0;
	while (c->name)
	{
		unsigned i = 0;
		char const *at_desc = NULL, *default_desc = NULL;
		int default_value = -1;

		printf("%-25s = ", c->name);

		if (get_defined(parm->is_defined, ndx) == 0)
		{
			if (c->offset == offsetof(parm_t, scan_options.general) ||
				c->offset == offsetof(parm_t, scan_options.parse) ||
				c->offset == offsetof(parm_t, scan_options.heuristic) ||
				c->offset == offsetof(parm_t, scan_options.mail) ||
				c->offset == offsetof(parm_t, scan_options.dev) ||
				c->offset == offsetof(parm_t, load_options))
					get_bool(parm, c, &at_desc, &default_value);
			else if (c->assign_fn == assign_dir)
				default_desc = get_default_dir(c->offset);

			fputs("not set", stdout);
		}
		else if (c->assign_fn == assign_char || c->assign_fn == assign_option)
		{
			printf("%d", get_bool(parm, c, &at_desc, &default_value));
		}
		else if (c->assign_fn == assign_dir)
		{
			char const * const p = get_dir(parm, c);
			fputs(p? p: "NULL", stdout);
			default_desc = get_default_dir(c->offset);
		}
		else if (c->assign_fn == assign_array)
		{
			charray const *a = PARM_PTR(charray const*);
			if (a == NULL)
				fputs("NULL", stdout);
			else
			{
				if (c->descr && c->descr[0])
					printf(" (%s)", c->descr);
				fputc('\n', stdout);
				for (; i < a->count; ++i)
					printf("%27d %s\n", i, a->array[i]);

				i = 1;
			}
		}
		else if (c->assign_fn == assign_case)
		{
			printf("%s", get_case(parm, c));
		}
		else
		{
			uintmax_t const v = get_int(parm, c);
			printf("%" PRIdMAX " (0x%" PRIxMAX ")", v, v);
		}

		if (i == 0)
		{
			// load and scan values already have at_desc at this point
			if (at_desc == NULL && c->flag != NO_FLAG)
				at_desc = "engine";

			if ((c->descr && c->descr[0]) || at_desc ||
				default_value >= 0 || default_desc)
			{
				fputs(" (", stdout);
				if (c->descr && c->descr[0])
				{
					fputs(c->descr, stdout);
					if (default_value >= 0 || default_desc || at_desc)
						fputs(", ", stdout);
				}
				if (at_desc)
				{
					fputs(at_desc, stdout);
					if (default_value >= 0 || default_desc)
						fputs(", ", stdout);
				}
				if (default_value >= 0)
					printf("default %d", default_value);
				else if (default_desc)
					printf("default %s", default_desc);
				fputc(')', stdout);
			}
			fputc('\n', stdout);
		}
		++c;
		++ndx;
	}
}

#endif // !defined NO_CLAMAV_FUNCTIONS


#if defined TEST_ARRAY
// gcc -std=gnu99 -DTEST_ARRAY -DAVFILTER_DEBUG -I.. -W -Wall -O0 -g3 -o tarray parm.c log.c
int main(int argc, char *argv[])
{
	parm_t parm;
	char *p = NULL;

	config_conf const testconf =
		{"test", "test", assign_array, 0, 0, sizeof parm.test};

	memset(&parm, 0, sizeof parm);

	for (int i = 1; i < argc; ++i)
	{
		// possibly force realloc to yield a new address
		free(p);
		if (isupper(*(unsigned char*)argv[i])) p = strdup(argv[i]);
		else p = NULL;

		if (assign_array(&parm, &testconf, argv[i]))
			break;

		charray *test = parm.test;
		printf("count=%zu, alloc=%zu, strsize=%zu, stralloc=%zu\n"
			"start=%p, first=%p (diff=%tu), end=%p (total size=%tu)\n",
			test->count, test->alloc, test->strsize, test->stralloc,
			test, CHARRAY_FIRST_STRING(*test),
				CHARRAY_FIRST_STRING(*test) - (char*)test,
				CHARRAY_OFF_THE_END(*test),
				CHARRAY_OFF_THE_END(*test) - (char*)test);
		size_t len = 0;
		for (size_t j = 0; j < test->count; ++j)
		{
			printf("   array[%zu]=\"%s\"\n", j, test->array[j]);
			len += strlen(test->array[j]) + 1;
		}
		assert(len == test->strsize);
		printf("   available: %zu strings + %zu str space (total waste: %zu)\n\n",
			test->alloc - test->count, test->stralloc - test->strsize,
			sizeof(char*)*(test->alloc - test->count) + test->stralloc - test->strsize);
	}

	free(p);
	free(parm.test);

	return 0;
}
#endif

#if defined TEST_DIRECTORY
// gcc -DTEST_DIRECTORY -DAVFILTER_DEBUG -I.. -W -Wall -O0 -g parm.c log.c
int main(int argc, char *argv[])
{
	for (int i = 1; i < argc; i += 2)
	{
		char const *name = i + 1 >= argc? "save_virus": argv[i+1];
		char *value = read_directory(argv[i], name);
		printf("%s = %s\n", name, value);
		free(value);
	}
	return 0;
}
#endif
